<?php

// ------------------------------------------------------------
// Contains message info
// ------------------------------------------------------------
abstract class MessageInfo
{
  public $JsonType;
  public $JsonVersion;
  public $OrderId;
}

// ------------------------------------------------------------
// Contains parsed data
// ------------------------------------------------------------
class Device extends MessageInfo
{
  public $Rssi;
  public $TimeNow;
  public $IsAsync;
  public $Sn;
  public $Desc;
  public $Kind;
  public $Time;
  public $IsTimezoneSet;
  public $Timezone;
  public $IntervalPowerMode;
  public $IntervalBatteryMode;
  public $RstOk;
  public $BufferedModeMaxMessagesCnt;
  public $AStateMask;
  public $AState;
  public $NConf;
  public $AlarmsEvalOffDueTimeLimit;
  public $CustomerUid;
  public $BatteryPercentage;
  public $Channels; // Array of Channel objects. Bounds: 1..8
}

// ------------------------------------------------------------
// Base class for channel data.
// Objects of this types are items in Device->Channels array.
// ------------------------------------------------------------
abstract class Channel
{
  public $Number;
  public $Quantity;
  public $Alarm1;
  public $Alarm2;
  public $Alarm1On;
  public $Alarm2On;
}

// ------------------------------------------------------------
// Analog channel data
// ------------------------------------------------------------
class AnalogChannel extends Channel
{
  public $Unit;
  public $DecimalPlaces;
  public $Value;
  public $IsValueCorrect;
  public $ValueError;
  public $Alarm1Limit;
  public $Alarm2Limit;
  public $Alarm1IsHigherThenLimit;
  public $Alarm2IsHigherThenLimit;
}

// ------------------------------------------------------------
// Binary channel data
// ------------------------------------------------------------
class BinaryChannel extends Channel
{
  public $BinDescs0;
  public $BinDescs1;
  public $State;
  public $Alarm1ActiveOnStateInHigh;
  public $Alarm2ActiveOnStateInHigh;
}

// ------------------------------------------------------------
// Error codes constants
// ------------------------------------------------------------
const ERROR_CODE_UNKNOWN = 1;
const ERROR_CODE_JSON_DECODE_FAILED = 2;
const ERROR_CODE_DUPLICITE_CHANEL_NUMBER = 3;
const ERROR_CODE_PROCESING_OF_PARSED_DATA_FAILED = 4;
const ERROR_CODE_UNKNOWN_JSON_TYPE = 5;
const ERROR_CODE_UNKNOWN_JSON_VERSION = 6;
const ERROR_CODE_IS_NOT_ARRAY = 11;
const ERROR_CODE_ARRAY_OUT_OF_BOUNDS = 12;
const ERROR_CODE_IS_NOT_INT = 13;
const ERROR_CODE_INT_OUT_OF_BOUNDS = 14;
const ERROR_CODE_IS_NOT_STRING = 15;
const ERROR_CODE_STRING_OUT_OF_BOUNDS = 16;
const ERROR_CODE_DATETIME_OUT_OF_BOUNDS = 17;
const ERROR_CODE_DATETIME_BAD_FORMAT = 18;
const ERROR_CODE_FLOAT_BAD_HEX_FORMAT = 19;
const ERROR_CODE_FLOAT_ERROR_NOT_ALLOWED = 20;

// ------------------------------------------------------------
// Exception object that holds information about parse error.
// Contains error code and four indexes detail to locate item in JSON data that caused the error.
// ------------------------------------------------------------
class ParseException extends Exception
{
  public $Arr0Idx;
  public $Arr1Idx;
  public $Arr2Idx;
  public $Arr3Idx;

  public function __construct($message, $code = ERROR_CODE_UNKNOWN, $Arr0Idx = -1, $Arr1Idx = -1, $Arr2Idx = -1, $Arr3Idx = -1, Exception $previous = null)
  {
    parent::__construct($message, $code, $previous);
    $this->Arr0Idx = $Arr0Idx;
    $this->Arr1Idx = $Arr1Idx;
    $this->Arr2Idx = $Arr2Idx;
    $this->Arr3Idx = $Arr3Idx;
  }

  public function GetJSONShortDetail()
  {
    return "[".$this->code.",".$this->Arr0Idx.",".$this->Arr1Idx.",".$this->Arr2Idx.",".$this->Arr3Idx."]";
  }
  public function ToString()
  {
    $errorCodeStr = "Unknown error code.";
    switch ($this->code)
    {
      case ERROR_CODE_UNKNOWN                         : $errorCodeStr = "(".ERROR_CODE_UNKNOWN.") Unknown error."; break;
      case ERROR_CODE_JSON_DECODE_FAILED              : $errorCodeStr = "(".ERROR_CODE_JSON_DECODE_FAILED.") JSON decode failed."; break;
      case ERROR_CODE_DUPLICITE_CHANEL_NUMBER         : $errorCodeStr = "(".ERROR_CODE_DUPLICITE_CHANEL_NUMBER.") Duplicite channel number found."; break;
      case ERROR_CODE_PROCESING_OF_PARSED_DATA_FAILED : $errorCodeStr = "(".ERROR_CODE_PROCESING_OF_PARSED_DATA_FAILED.") Processing of parsed data failed."; break;
      case ERROR_CODE_UNKNOWN_JSON_TYPE               : $errorCodeStr = "(".ERROR_CODE_UNKNOWN_JSON_TYPE.") Unknown JSON type found."; break;
      case ERROR_CODE_UNKNOWN_JSON_VERSION            : $errorCodeStr = "(".ERROR_CODE_UNKNOWN_JSON_VERSION.") Unknown JSON version found."; break;
      case ERROR_CODE_IS_NOT_ARRAY                    : $errorCodeStr = "(".ERROR_CODE_IS_NOT_ARRAY.") Argument is not array."; break;
      case ERROR_CODE_ARRAY_OUT_OF_BOUNDS             : $errorCodeStr = "(".ERROR_CODE_ARRAY_OUT_OF_BOUNDS.") Argument (array) is out of bounds."; break;
      case ERROR_CODE_IS_NOT_INT                      : $errorCodeStr = "(".ERROR_CODE_IS_NOT_INT.") Argument is not int."; break;
      case ERROR_CODE_INT_OUT_OF_BOUNDS               : $errorCodeStr = "(".ERROR_CODE_INT_OUT_OF_BOUNDS.") Argument (int) is out of bounds."; break;
      case ERROR_CODE_IS_NOT_STRING                   : $errorCodeStr = "(".ERROR_CODE_IS_NOT_STRING.") Argument is not string."; break;
      case ERROR_CODE_STRING_OUT_OF_BOUNDS            : $errorCodeStr = "(".ERROR_CODE_STRING_OUT_OF_BOUNDS.") Argument (string) is out of bounds."; break;
      case ERROR_CODE_DATETIME_OUT_OF_BOUNDS          : $errorCodeStr = "(".ERROR_CODE_DATETIME_OUT_OF_BOUNDS.") Argument (datetime) is out of bounds."; break;
      case ERROR_CODE_DATETIME_BAD_FORMAT             : $errorCodeStr = "(".ERROR_CODE_DATETIME_BAD_FORMAT.") Argument (datetime) is of bad format."; break;
      case ERROR_CODE_FLOAT_BAD_HEX_FORMAT            : $errorCodeStr = "(".ERROR_CODE_FLOAT_BAD_HEX_FORMAT.") Argument (float) is of bad hex format."; break;
      case ERROR_CODE_FLOAT_ERROR_NOT_ALLOWED         : $errorCodeStr = "(".ERROR_CODE_FLOAT_ERROR_NOT_ALLOWED.") Argument (float) error value is not allowed."; break;
    }
    return "Error code: " . $errorCodeStr . " Short detail: " . $this->GetJSONShortDetail() . " Message: " . $this->getMessage();
  }
}

// ------------------------------------------------------------
// Helper class for parsing POST data input
// Contains only these two public functions for use from outside:
// Parse($input) - Parse POST data input
// ToString(Device $deviceObj) - returns user friendly information about $deviceObj object
// ------------------------------------------------------------
class DeviceHelper
{
  //stare parsovani pro FW 0.0.90. Vola se v Parse, kdyz selze Parsovani 1. dvou polozek JSON (verze a typ)
  public static function Parse($input)
  {
    DeviceHelper::ParseString($input, 1, 10000);
    $jsonArr = json_decode($input);
    if (json_last_error() != JSON_ERROR_NONE)
    {
      //failed, so next attemt will encode input into utf8 before calling json decode
      $orig_err_str = "JSON error: " . json_last_error_msg();
      $jsonArr = json_decode(utf8_encode($input));//ulogger string format is iso 8859-1 expected
      if (json_last_error() != JSON_ERROR_NONE)
        throw new ParseException($orig_err_str, ERROR_CODE_JSON_DECODE_FAILED);
    }
    //Currently only JSON type = 1 and JSON version = 1, 2, 3, 4 exists. When other types and versions came, further code and decisions will have to be implemented!
    //info about version: see ULoggerJsonDoc.txt

    $device = null;
    $jsonType = null;
    $jsonVer = null;
    try
    {
      DeviceHelper::CheckIsArray($jsonArr, 2, 999);
      $jsonType = DeviceHelper::ParseInt($jsonArr[0], 0, 255, 0);
      $jsonVer = DeviceHelper::ParseInt($jsonArr[1], 0, 255, 1);
    }
    catch (Exception $e)
    {
      $excBckp = $e;
      try
      {
        //volani 0.0.90 parsovani
        return DeviceHelper::Parse_FW_0_0_90($input);
      }
      catch (Exception $e2)
      {
        //vyjimka Parse fw 90 se zahazuje :-( o duvodu chyby parsovani fw 90 se caller nedozvi :-(
        throw $excBckp;
      }
    }
    if ($jsonType == 1)
    {
      if (
           ($jsonVer == 1) || 
           ($jsonVer == 2) || 
           ($jsonVer == 3) ||
           ($jsonVer == 4) ||
           ($jsonVer == 5) ||
           ($jsonVer == 6)
      )
      {
        $device = new Device();
        $device->JsonVersion = $jsonVer;
      }
      else
        throw new ParseException("JSON Type = " . $jsonType . ", JSON Version = " . $jsonVer, ERROR_CODE_UNKNOWN_JSON_VERSION);
      $device->JsonType = $jsonType;
    }
    else
      throw new ParseException("JSON Type = " . $jsonType, ERROR_CODE_UNKNOWN_JSON_TYPE);

    if (($jsonVer == 1) || ($jsonVer == 2))
      return DeviceHelper::ParseType1Version1and2($device, $jsonArr);
    else if ($jsonVer == 3)
      return DeviceHelper::ParseType1Version3($device, $jsonArr);
    else if ($jsonVer == 4)
      return DeviceHelper::ParseType1Version4($device, $jsonArr);
    else if ($jsonVer == 5)
      return DeviceHelper::ParseType1Version5($device, $jsonArr);
    else
      return DeviceHelper::ParseType1Version6($device, $jsonArr);

  }
  public static function ToString(Device $deviceObj)
  {
    $deviceStr =
      "JsonType = "                   . $deviceObj->JsonType . "\r\n" .
      "JsonVersion = "                . $deviceObj->JsonVersion . "\r\n" .
      "OrderId = "                    . $deviceObj->OrderId . "\r\n" .
      "Rssi = "                       . $deviceObj->Rssi . "\r\n" .
      "TimeNow = "                    . $deviceObj->TimeNow->format("Y-m-d\TH:i:s") . "\r\n" .
      "SerialNumber = "               . $deviceObj->Sn . "\r\n" .
      "Desc = "                       . "\"" . utf8_decode($deviceObj->Desc) . "\"" . "\r\n" .
      "Kind = "                       . $deviceObj->Kind . "\r\n" .
      "Time = "                       . $deviceObj->Time->format("Y-m-d\TH:i:s") . "\r\n" .
      "IsTimezoneSet = "              . ($deviceObj->IsTimezoneSet ? "true" : "false") . "\r\n" .
      "Timezone = "                   . ($deviceObj->IsTimezoneSet ?
                                          ($deviceObj->Timezone >= 0 ? '+' : '-') 
                                          . sprintf('%02d', floor(abs($deviceObj->Timezone) / 60)) . ':' 
                                          . sprintf('%02d', floor(abs($deviceObj->Timezone) % 60)) 
                                        : "---") . "\r\n" .
      "IntervalPowerMode = "          . $deviceObj->IntervalPowerMode . "\r\n" .
      "IntervalBatteryMode = "        . $deviceObj->IntervalBatteryMode . "\r\n" .
      "RstOk = "                      . ($deviceObj->RstOk ? "true" : "false") . "\r\n" .
      "BufferedModeMaxMessagesCnt = " . $deviceObj->BufferedModeMaxMessagesCnt . "\r\n" .
      "IsAsync = "                    . ($deviceObj->IsAsync ? "true" : "false") . "\r\n" .
      "AStateMask = "                 . $deviceObj->AStateMask . "\r\n" .
      DeviceHelper::AStateToString($deviceObj->AStateMask) .
      "AState = "                     . $deviceObj->AState . "\r\n" .
      DeviceHelper::AStateToString($deviceObj->AState) .
      "NConf = "                      . $deviceObj->NConf . "\r\n" .
      "AlarmsEvalOffDueTimeLimit = "  . ($deviceObj->AlarmsEvalOffDueTimeLimit ? "true" : "false") . "\r\n" .
      "CustomerUid = "                . "\"" . $deviceObj->CustomerUid . "\"" . "\r\n" .
      "BatteryPercentage = "          . $deviceObj->BatteryPercentage . "\r\n" .
      "Channels:\r\n";
    $channels = "";
    for ($i = 0; $i < count($deviceObj->Channels); $i++)
    {
      $channels .= "  Channel number = " . $deviceObj->Channels[$i]->Number . "\r\n";
      $channels .= "    Quantity = " . "\"" . utf8_decode($deviceObj->Channels[$i]->Quantity) . "\"" . "\r\n";

      if (is_a($deviceObj->Channels[$i], 'AnalogChannel'))
      {
        $channels .= "    Value = " . round($deviceObj->Channels[$i]->Value, 2) . "\r\n";
        $channels .= "    IsValueCorrect = " . ($deviceObj->Channels[$i]->IsValueCorrect ? "true" : "false") . "\r\n";
        $channels .= "    ValueError = " . $deviceObj->Channels[$i]->ValueError . "\r\n";
        $channels .= "    Unit = " . "\"" . utf8_decode($deviceObj->Channels[$i]->Unit) . "\"" . "\r\n";
        $channels .= "    DecimalPlaces = " . $deviceObj->Channels[$i]->DecimalPlaces . "\r\n";
        $channels .= "    Alarm1 = " . ($deviceObj->Channels[$i]->Alarm1 ? "true" : "false") . "\r\n";
        $channels .= "    Alarm2 = " . ($deviceObj->Channels[$i]->Alarm2 ? "true" : "false") . "\r\n";
        $channels .= "    Alarm1On = " . ($deviceObj->Channels[$i]->Alarm1On ? "true" : "false") . "\r\n";
        $channels .= "    Alarm2On = " . ($deviceObj->Channels[$i]->Alarm2On ? "true" : "false") . "\r\n";
        $channels .= "    Alarm1Limit = " . round($deviceObj->Channels[$i]->Alarm1Limit, 2) . "\r\n";
        $channels .= "    Alarm2Limit = " . round($deviceObj->Channels[$i]->Alarm2Limit, 2) . "\r\n";
        $channels .= "    Alarm1IsHigherThenLimit = " . ($deviceObj->Channels[$i]->Alarm1IsHigherThenLimit ? "true" : "false") . "\r\n";
        $channels .= "    Alarm2IsHigherThenLimit = " . ($deviceObj->Channels[$i]->Alarm2IsHigherThenLimit ? "true" : "false") . "\r\n";
      }
      else if (is_a($deviceObj->Channels[$i], 'BinaryChannel'))
      {
        $channels .= "    State = " . ($deviceObj->Channels[$i]->State ? "true" : "false") . "\r\n";
        $channels .= "    BinDescs0 = " . "\"" . utf8_decode($deviceObj->Channels[$i]->BinDescs0) . "\"" . "\r\n";
        $channels .= "    BinDescs1 = " . "\"" . utf8_decode($deviceObj->Channels[$i]->BinDescs1) . "\"" . "\r\n";
        $channels .= "    Alarm1 = " . ($deviceObj->Channels[$i]->Alarm1 ? "true" : "false") . "\r\n";
        $channels .= "    Alarm2 = " . ($deviceObj->Channels[$i]->Alarm2 ? "true" : "false") . "\r\n";
        $channels .= "    Alarm1On = " . ($deviceObj->Channels[$i]->Alarm1On ? "true" : "false") . "\r\n";
        $channels .= "    Alarm2On = " . ($deviceObj->Channels[$i]->Alarm2On ? "true" : "false") . "\r\n";
        $channels .= "    Alarm1ActiveOnStateInHigh = " . ($deviceObj->Channels[$i]->Alarm1ActiveOnStateInHigh ? "true" : "false") . "\r\n";
        $channels .= "    Alarm2ActiveOnStateInHigh = " . ($deviceObj->Channels[$i]->Alarm2ActiveOnStateInHigh ? "true" : "false") . "\r\n";
      }
      else
        throw new ErrorException("DeviceHelper::ToString(...): Unexpected type of channel.");
    }
    return $deviceStr . $channels;
  }
  public static function ToStringAStateOnly(Device $deviceObj)
  {
    return
      "AStateMask, AState = " . $deviceObj->AStateMask . ", " . $deviceObj->AState . "\r\n" .
      DeviceHelper::AStateWithMaskToString($deviceObj->AStateMask, $deviceObj->AState);
  }

  private static function Parse_FW_0_0_90($input)
  {
    DeviceHelper::ParseString($input, 1, 10000);
    $jsonArr = json_decode(utf8_encode($input));//ulogger string format is iso 8859-1 expected
    if (json_last_error() != JSON_ERROR_NONE)
      throw new ParseException("JSON error: " . json_last_error_msg(), ERROR_CODE_JSON_DECODE_FAILED);

    $device = new Device();

    DeviceHelper::CheckIsArray($jsonArr, 9, 9);

    $device->Sn = DeviceHelper::ParseInt($jsonArr[0], 0, 99999999, 0);
    $device->Desc = DeviceHelper::ParseString($jsonArr[1], 0, 32, 1);
    $device->Kind = DeviceHelper::ParseInt($jsonArr[2], 0, 255, 2);
    $device->Time = DeviceHelper::ParseDatetime($jsonArr[3], 3);
    $timezone15minsTmp = DeviceHelper::ParseInt($jsonArr[4], -128, 127, 4);
    $device->Timezone = $timezone15minsTmp * 15;
    $device->IsTimezoneSet = $timezone15minsTmp != -128;
    $device->IntervalPowerMode = DeviceHelper::ParseInt($jsonArr[9], 0, 604800, 9);
    $device->IntervalBatteryMode = $device->IntervalPowerMode;
    $device->RstOk = false;
    $device->Flags = DeviceHelper::ParseInt($jsonArr[6], 0, 65535, 5);
    $device->CustomerUid = DeviceHelper::ParseString($jsonArr[7], 0, 16, 6);
    $device->Channels = array();

    $chsIdx = 8;
    $jsonChs = $jsonArr[$chsIdx];
    DeviceHelper::CheckIsArray($jsonChs, 0, 8, $chsIdx);
    for ($i = 0; $i < count($jsonChs); $i++)
    {
      $jsonCh = $jsonChs[$i];
      DeviceHelper::CheckIsArray($jsonCh, 2, 9, $chsIdx, $i);

      $number = DeviceHelper::ParseInt($jsonCh[0], 1, 8, $chsIdx, $i, 0);
      $j = 0;
      $isDupl = false;
      while (($j < count($device->Channels)) && (!$isDupl))
      {
        if ($device->Channels[$j]->Number == $number)
          $isDupl = true;
        $j++;
      }
      if ($isDupl)
        throw new ParseException("", ERROR_CODE_DUPLICITE_CHANEL_NUMBER, $chsIdx, $i, 0);

      $isBin = DeviceHelper::parseBool($jsonCh[1], $chsIdx, $i, 1) == 1;
      if ($isBin)
        DeviceHelper::CheckIsArray($jsonCh, 7, 7, $chsIdx, $i);
      else
        DeviceHelper::CheckIsArray($jsonCh, 9, 9, $chsIdx, $i);
      $channel = $isBin ? $channel = new BinaryChannel() : $channel = new AnalogChannel();
      $channel->Number = $number;
      $channel->Quantity = DeviceHelper::ParseString($jsonCh[2], 0, 16, $chsIdx, $i, 2);
      if ($isBin)
      {
        DeviceHelper::CheckIsArray($jsonCh[3], 2, 2, $chsIdx, $i, 3);
        $channel->BinDescs0 = DeviceHelper::ParseString($jsonCh[3][0], 0, 16, $chsIdx, $i, 3, 0);
        $channel->BinDescs1 = DeviceHelper::ParseString($jsonCh[3][1], 0, 16, $chsIdx, $i, 3, 1);
        $channel->State = DeviceHelper::ParseBool($jsonCh[4], $chsIdx, $i, 4);
        DeviceHelper::CheckIsArray($jsonCh[5], 2, 2, $chsIdx, $i, 5);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[5][0], $chsIdx, $i, 5, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[5][1], $chsIdx, $i, 5, 1);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[6][0], 0, 2, $chsIdx, $i, 6, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[6][1], 0, 2, $chsIdx, $i, 6, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1ActiveOnStateInHigh = $tmp1 == 2 ? true : false;
        $channel->Alarm2ActiveOnStateInHigh = $tmp2 == 2 ? true : false;
      }
      else
      {
        $value = 0.0;
        $error = 0;
        $channel->IsValueCorrect = false;
        $channel->Value = 0.0;
        $channel->ValueError = 0;
        if (DeviceHelper::ParseHexFloat($jsonCh[3], $value, $error, $chsIdx, $i, 3))
        {
          $channel->Value = $value;
          $channel->IsValueCorrect = true;
        }
        else
          $channel->ValueError = $error;
        $channel->Unit = DeviceHelper::ParseString($jsonCh[4], 0, 8, $chsIdx, $i, 4);
        $channel->DecimalPlaces = DeviceHelper::ParseInt($jsonCh[5], 0, 10, $chsIdx, $i, 5);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[6][0], $chsIdx, $i, 6, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[6][1], $chsIdx, $i, 6, 1);
        DeviceHelper::CheckIsArray($jsonCh[7], 2, 2, $chsIdx, $i, 7);
        $channel->Alarm1Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][0], $chsIdx, $i, 7, 0);
        $channel->Alarm2Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][1], $chsIdx, $i, 7, 1);
        DeviceHelper::CheckIsArray($jsonCh[8], 2, 2, $chsIdx, $i, 8);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[8][0], 0, 2, $chsIdx, $i, 8, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[8][1], 0, 2, 8, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1IsHigherThenLimit = $tmp1 == 2 ? true : false;
        $channel->Alarm2IsHigherThenLimit = $tmp2 == 2 ? true : false;
      }
      array_push($device->Channels, $channel);
    }

    return $device;
  }
  private static function ParseType1Version1and2($device, $jsonArr)
  {
    DeviceHelper::CheckIsArray($jsonArr, 15, 15);
    $device->Rssi = 99;//not known or not detectable
    $device->TimeNow = DateTime::createFromFormat(DateTime::ISO8601, "2000-01-01T00:00:00Z"); //exists since version 6
    $device->OrderId = DeviceHelper::ParseInt($jsonArr[2], 0, 255, 2);
    $device->IsAsync = DeviceHelper::ParseBool($jsonArr[3], 3);
    $device->Sn = DeviceHelper::ParseInt($jsonArr[4], 0, 99999999, 4);
    $device->Desc = DeviceHelper::ParseString($jsonArr[5], 0, 32, 5);
    $device->Kind = DeviceHelper::ParseInt($jsonArr[6], 0, 255, 6);
    $device->Time = DeviceHelper::ParseDatetime($jsonArr[7], 7);
    $timezone15minsTmp = DeviceHelper::ParseInt($jsonArr[8], -128, 127, 8);
    $device->Timezone = $timezone15minsTmp * 15;
    $device->IsTimezoneSet = $timezone15minsTmp != -128;
    $device->IntervalPowerMode = DeviceHelper::ParseInt($jsonArr[9], 0, 604800, 9);
    $device->IntervalBatteryMode = $device->IntervalPowerMode;
    $device->RstOk = false;
    $device->AState = DeviceHelper::ParseInt($jsonArr[10], 0, 65535, 10);
    $device->NConf = DeviceHelper::ParseInt($jsonArr[11], 0, 255, 11);
    $device->AlarmsEvalOffDueTimeLimit = DeviceHelper::ParseBool($jsonArr[12], 12);
    $device->CustomerUid = DeviceHelper::ParseString($jsonArr[13], 0, 16, 13);
    $device->BatteryPercentage = 0;//existuje a6 od verze json 5
    $device->Channels = array();

    $chsIdx = 14;
    $jsonChs = $jsonArr[$chsIdx];
    DeviceHelper::CheckIsArray($jsonChs, 0, 8, $chsIdx);
    for ($i = 0; $i < count($jsonChs); $i++)
    {
      $jsonCh = $jsonChs[$i];
      DeviceHelper::CheckIsArray($jsonCh, 2, 9, $chsIdx, $i);

      $number = DeviceHelper::ParseInt($jsonCh[0], 1, 8, $chsIdx, $i, 0);
      $j = 0;
      $isDupl = false;
      while (($j < count($device->Channels)) && (!$isDupl))
      {
        if ($device->Channels[$j]->Number == $number)
          $isDupl = true;
        $j++;
      }
      if ($isDupl)
        throw new ParseException("", ERROR_CODE_DUPLICITE_CHANEL_NUMBER, $chsIdx, $i, 0);

      $isBin = DeviceHelper::parseBool($jsonCh[1], $chsIdx, $i, 1) == 1;
      if ($isBin)
        DeviceHelper::CheckIsArray($jsonCh, 7, 7, $chsIdx, $i);
      else
        DeviceHelper::CheckIsArray($jsonCh, 9, 9, $chsIdx, $i);
      $channel = $isBin ? $channel = new BinaryChannel() : $channel = new AnalogChannel();
      $channel->Number = $number;
      $channel->Quantity = DeviceHelper::ParseString($jsonCh[2], 0, 16, $chsIdx, $i, 2);
      if ($isBin)
      {
        DeviceHelper::CheckIsArray($jsonCh[3], 2, 2, $chsIdx, $i, 3);
        $channel->BinDescs0 = DeviceHelper::ParseString($jsonCh[3][0], 0, 16, $chsIdx, $i, 3, 0);
        $channel->BinDescs1 = DeviceHelper::ParseString($jsonCh[3][1], 0, 16, $chsIdx, $i, 3, 1);
        $channel->State = DeviceHelper::ParseBool($jsonCh[4], $chsIdx, $i, 4);
        DeviceHelper::CheckIsArray($jsonCh[5], 2, 2, $chsIdx, $i, 5);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[5][0], $chsIdx, $i, 5, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[5][1], $chsIdx, $i, 5, 1);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[6][0], 0, 2, $chsIdx, $i, 6, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[6][1], 0, 2, $chsIdx, $i, 6, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1ActiveOnStateInHigh = $tmp1 == 2 ? true : false;
        $channel->Alarm2ActiveOnStateInHigh = $tmp2 == 2 ? true : false;
      }
      else
      {
        $value = 0.0;
        $error = 0;
        $channel->IsValueCorrect = false;
        $channel->Value = 0.0;
        $channel->ValueError = 0;
        if (DeviceHelper::ParseHexFloat($jsonCh[3], $value, $error, $chsIdx, $i, 3))
        {
          $channel->Value = $value;
          $channel->IsValueCorrect = true;
        }
        else
          $channel->ValueError = $error;
        $channel->Unit = DeviceHelper::ParseString($jsonCh[4], 0, 8, $chsIdx, $i, 4);
        $channel->DecimalPlaces = DeviceHelper::ParseInt($jsonCh[5], 0, 10, $chsIdx, $i, 5);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[6][0], $chsIdx, $i, 6, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[6][1], $chsIdx, $i, 6, 1);
        DeviceHelper::CheckIsArray($jsonCh[7], 2, 2, $chsIdx, $i, 7);
        $channel->Alarm1Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][0], $chsIdx, $i, 7, 0);
        $channel->Alarm2Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][1], $chsIdx, $i, 7, 1);
        DeviceHelper::CheckIsArray($jsonCh[8], 2, 2, $chsIdx, $i, 8);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[8][0], 0, 2, $chsIdx, $i, 8, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[8][1], 0, 2, $chsIdx, $i, 8, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1IsHigherThenLimit = $tmp1 == 2 ? true : false;
        $channel->Alarm2IsHigherThenLimit = $tmp2 == 2 ? true : false;
      }
      array_push($device->Channels, $channel);
    }
    return $device;
  }
  private static function ParseType1Version3($device, $jsonArr)
  {
    DeviceHelper::CheckIsArray($jsonArr, 17, 17);
    $device->Rssi = 99;//not known or not detectable
    $device->TimeNow = DateTime::createFromFormat(DateTime::ISO8601, "2000-01-01T00:00:00Z"); //exists since version 6
    $device->OrderId = DeviceHelper::ParseInt($jsonArr[2], 0, 255, 2);
    $device->IsAsync = DeviceHelper::ParseBool($jsonArr[3], 3);
    $device->Sn = DeviceHelper::ParseInt($jsonArr[4], 0, 99999999, 4);
    $device->Desc = DeviceHelper::ParseString($jsonArr[5], 0, 32, 5);
    $device->Kind = DeviceHelper::ParseInt($jsonArr[6], 0, 255, 6);
    $device->Time = DeviceHelper::ParseDatetime($jsonArr[7], 7);
    $timezone15minsTmp = DeviceHelper::ParseInt($jsonArr[8], -128, 127, 8);
    $device->Timezone = $timezone15minsTmp * 15;
    $device->IsTimezoneSet = $timezone15minsTmp != -128;
    $device->IntervalPowerMode = DeviceHelper::ParseInt($jsonArr[9], 0, 604800, 9);
    $device->IntervalBatteryMode = $device->IntervalPowerMode;
    $device->RstOk = false;
    $device->BufferedModeMaxMessagesCnt = DeviceHelper::ParseInt($jsonArr[10], 0, 255, 10);
    $device->AStateMask = DeviceHelper::ParseInt($jsonArr[11], 0, 65535, 11);
    $device->AState = DeviceHelper::ParseInt($jsonArr[12], 0, 65535, 12);
    $device->NConf = DeviceHelper::ParseInt($jsonArr[13], 0, 255, 13);
    $device->AlarmsEvalOffDueTimeLimit = DeviceHelper::ParseBool($jsonArr[14], 14);
    $device->CustomerUid = DeviceHelper::ParseString($jsonArr[15], 0, 16, 15);
    $device->BatteryPercentage = 0;//existuje a6 od verze json 5
    $device->Channels = array();

    $chsIdx = 16;
    $jsonChs = $jsonArr[$chsIdx];
    DeviceHelper::CheckIsArray($jsonChs, 0, 8, $chsIdx);
    for ($i = 0; $i < count($jsonChs); $i++)
    {
      $jsonCh = $jsonChs[$i];
      DeviceHelper::CheckIsArray($jsonCh, 2, 9, $chsIdx, $i);

      $number = DeviceHelper::ParseInt($jsonCh[0], 1, 8, $chsIdx, $i, 0);
      $j = 0;
      $isDupl = false;
      while (($j < count($device->Channels)) && (!$isDupl))
      {
        if ($device->Channels[$j]->Number == $number)
          $isDupl = true;
        $j++;
      }
      if ($isDupl)
        throw new ParseException("", ERROR_CODE_DUPLICITE_CHANEL_NUMBER, $chsIdx, $i, 0);

      $isBin = DeviceHelper::parseBool($jsonCh[1], $chsIdx, $i, 1) == 1;
      if ($isBin)
        DeviceHelper::CheckIsArray($jsonCh, 7, 7, $chsIdx, $i);
      else
        DeviceHelper::CheckIsArray($jsonCh, 9, 9, $chsIdx, $i);
      $channel = $isBin ? $channel = new BinaryChannel() : $channel = new AnalogChannel();
      $channel->Number = $number;
      $channel->Quantity = DeviceHelper::ParseString($jsonCh[2], 0, 16, $chsIdx, $i, 2);
      if ($isBin)
      {
        DeviceHelper::CheckIsArray($jsonCh[3], 2, 2, $chsIdx, $i, 3);
        $channel->BinDescs0 = DeviceHelper::ParseString($jsonCh[3][0], 0, 16, $chsIdx, $i, 3, 0);
        $channel->BinDescs1 = DeviceHelper::ParseString($jsonCh[3][1], 0, 16, $chsIdx, $i, 3, 1);
        $channel->State = DeviceHelper::ParseBool($jsonCh[4], $chsIdx, $i, 4);
        DeviceHelper::CheckIsArray($jsonCh[5], 2, 2, $chsIdx, $i, 5);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[5][0], $chsIdx, $i, 5, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[5][1], $chsIdx, $i, 5, 1);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[6][0], 0, 2, $chsIdx, $i, 6, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[6][1], 0, 2, $chsIdx, $i, 6, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1ActiveOnStateInHigh = $tmp1 == 2 ? true : false;
        $channel->Alarm2ActiveOnStateInHigh = $tmp2 == 2 ? true : false;
      }
      else
      {
        $value = 0.0;
        $error = 0;
        $channel->IsValueCorrect = false;
        $channel->Value = 0.0;
        $channel->ValueError = 0;
        if (DeviceHelper::ParseHexFloat($jsonCh[3], $value, $error, $chsIdx, $i, 3))
        {
          $channel->Value = $value;
          $channel->IsValueCorrect = true;
        }
        else
          $channel->ValueError = $error;
        $channel->Unit = DeviceHelper::ParseString($jsonCh[4], 0, 8, $chsIdx, $i, 4);
        $channel->DecimalPlaces = DeviceHelper::ParseInt($jsonCh[5], 0, 10, $chsIdx, $i, 5);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[6][0], $chsIdx, $i, 6, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[6][1], $chsIdx, $i, 6, 1);
        DeviceHelper::CheckIsArray($jsonCh[7], 2, 2, $chsIdx, $i, 7);
        $channel->Alarm1Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][0], $chsIdx, $i, 7, 0);
        $channel->Alarm2Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][1], $chsIdx, $i, 7, 1);
        DeviceHelper::CheckIsArray($jsonCh[8], 2, 2, $chsIdx, $i, 8);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[8][0], 0, 2, $chsIdx, $i, 8, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[8][1], 0, 2, $chsIdx, $i, 8, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1IsHigherThenLimit = $tmp1 == 2 ? true : false;
        $channel->Alarm2IsHigherThenLimit = $tmp2 == 2 ? true : false;
      }
      array_push($device->Channels, $channel);
    }
    return $device;
  }
  private static function ParseType1Version4($device, $jsonArr)
  {
    DeviceHelper::CheckIsArray($jsonArr, 20, 20);
    $device->Rssi = DeviceHelper::ParseInt($jsonArr[2], 0, 99, 2);
    $device->TimeNow = DateTime::createFromFormat(DateTime::ISO8601, "2000-01-01T00:00:00Z"); //exists since version 6
    $device->OrderId = DeviceHelper::ParseInt($jsonArr[3], 0, 255, 3);
    $device->IsAsync = DeviceHelper::ParseBool($jsonArr[4], 4);
    $device->Sn = DeviceHelper::ParseInt($jsonArr[5], 0, 99999999, 5);
    $device->Desc = DeviceHelper::ParseString($jsonArr[6], 0, 32, 6);
    $device->Kind = DeviceHelper::ParseInt($jsonArr[7], 0, 255, 7);

    $device->Time = DeviceHelper::ParseDatetime($jsonArr[8], 8);//JP: difference against comet cloud parser: here device time as is, in comet cloud timer time is converted to UTC
    $timezone15minsTmp = DeviceHelper::ParseInt($jsonArr[9], -128, 127, 9);
    $device->Timezone = $timezone15minsTmp * 15;
    $device->IsTimezoneSet = $timezone15minsTmp != -128;

    $device->IntervalPowerMode = DeviceHelper::ParseInt($jsonArr[10], 0, 604800, 10);
    $device->IntervalBatteryMode = DeviceHelper::ParseInt($jsonArr[11], 0, 604800, 11);
    $device->RstOk = DeviceHelper::ParseBool($jsonArr[12], 12);
    $device->BufferedModeMaxMessagesCnt = DeviceHelper::ParseInt($jsonArr[13], 0, 255, 13);
    $device->AStateMask = DeviceHelper::ParseInt($jsonArr[14], 0, 65535, 14);
    $device->AState = DeviceHelper::ParseInt($jsonArr[15], 0, 65535, 15);
    $device->NConf = DeviceHelper::ParseInt($jsonArr[16], 0, 255, 16);
    $device->AlarmsEvalOffDueTimeLimit = DeviceHelper::ParseBool($jsonArr[17], 17);
    $device->CustomerUid = DeviceHelper::ParseString($jsonArr[18], 0, 16, 18);
    $device->BatteryPercentage = 0;//existuje a6 od verze json 5
    $device->Channels = array();

    $chsIdx = 19;
    $jsonChs = $jsonArr[$chsIdx];
    DeviceHelper::CheckIsArray($jsonChs, 0, 8, $chsIdx);
    for ($i = 0; $i < count($jsonChs); $i++)
    {
      $jsonCh = $jsonChs[$i];
      DeviceHelper::CheckIsArray($jsonCh, 2, 9, $chsIdx, $i);

      $number = DeviceHelper::ParseInt($jsonCh[0], 1, 8, $chsIdx, $i, 0);
      $j = 0;
      $isDupl = false;
      while (($j < count($device->Channels)) && (!$isDupl))
      {
        if ($device->Channels[$j]->Number == $number)
          $isDupl = true;
        $j++;
      }
      if ($isDupl)
        throw new ParseException("", ERROR_CODE_DUPLICITE_CHANEL_NUMBER, $chsIdx, $i, 0);

      $isBin = DeviceHelper::parseBool($jsonCh[1], $chsIdx, $i, 1) == 1;
      if ($isBin)
        DeviceHelper::CheckIsArray($jsonCh, 7, 7, $chsIdx, $i);
      else
        DeviceHelper::CheckIsArray($jsonCh, 9, 9, $chsIdx, $i);
      $channel = $isBin ? $channel = new BinaryChannel() : $channel = new AnalogChannel();
      $channel->Number = $number;
      $channel->Quantity = DeviceHelper::ParseString($jsonCh[2], 0, 16, $chsIdx, $i, 2);
      if ($isBin)
      {
        DeviceHelper::CheckIsArray($jsonCh[3], 2, 2, $chsIdx, $i, 3);
        $channel->BinDescs0 = DeviceHelper::ParseString($jsonCh[3][0], 0, 16, $chsIdx, $i, 3, 0);
        $channel->BinDescs1 = DeviceHelper::ParseString($jsonCh[3][1], 0, 16, $chsIdx, $i, 3, 1);
        $channel->State = DeviceHelper::ParseBool($jsonCh[4], $chsIdx, $i, 4);
        DeviceHelper::CheckIsArray($jsonCh[5], 2, 2, $chsIdx, $i, 5);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[5][0], $chsIdx, $i, 5, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[5][1], $chsIdx, $i, 5, 1);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[6][0], 0, 2, $chsIdx, $i, 6, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[6][1], 0, 2, $chsIdx, $i, 6, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1ActiveOnStateInHigh = $tmp1 == 2 ? true : false;
        $channel->Alarm2ActiveOnStateInHigh = $tmp2 == 2 ? true : false;
      }
      else
      {
        $value = 0.0;
        $error = 0;
        $channel->IsValueCorrect = false;
        $channel->Value = 0.0;
        $channel->ValueError = 0;
        if (DeviceHelper::ParseHexFloat($jsonCh[3], $value, $error, $chsIdx, $i, 3))
        {
          $channel->Value = $value;
          $channel->IsValueCorrect = true;
        }
        else
          $channel->ValueError = $error;
        $channel->Unit = DeviceHelper::ParseString($jsonCh[4], 0, 8, $chsIdx, $i, 4);
        $channel->DecimalPlaces = DeviceHelper::ParseInt($jsonCh[5], 0, 10, $chsIdx, $i, 5);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[6][0], $chsIdx, $i, 6, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[6][1], $chsIdx, $i, 6, 1);
        DeviceHelper::CheckIsArray($jsonCh[7], 2, 2, $chsIdx, $i, 7);
        $channel->Alarm1Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][0], $chsIdx, $i, 7, 0);
        $channel->Alarm2Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][1], $chsIdx, $i, 7, 1);
        DeviceHelper::CheckIsArray($jsonCh[8], 2, 2, $chsIdx, $i, 8);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[8][0], 0, 2, $chsIdx, $i, 8, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[8][1], 0, 2, $chsIdx, $i, 8, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1IsHigherThenLimit = $tmp1 == 2 ? true : false;
        $channel->Alarm2IsHigherThenLimit = $tmp2 == 2 ? true : false;
      }
      array_push($device->Channels, $channel);
    }
    return $device;
  }
  private static function ParseType1Version5($device, $jsonArr)
  {
    DeviceHelper::CheckIsArray($jsonArr, 21, 21);
    $device->Rssi = DeviceHelper::ParseInt($jsonArr[2], 0, 99, 2);
    $device->TimeNow = DateTime::createFromFormat(DateTime::ISO8601, "2000-01-01T00:00:00Z"); //exists since version 6
    $device->OrderId = DeviceHelper::ParseInt($jsonArr[3], 0, 255, 3);
    $device->IsAsync = DeviceHelper::ParseBool($jsonArr[4], 4);
    $device->Sn = DeviceHelper::ParseInt($jsonArr[5], 0, 99999999, 5);
    $device->Desc = DeviceHelper::ParseString($jsonArr[6], 0, 32, 6);
    $device->Kind = DeviceHelper::ParseInt($jsonArr[7], 0, 255, 7);

    $device->Time = DeviceHelper::ParseDatetime($jsonArr[8], 8);//JP: difference against comet cloud parser: here device time as is, in comet cloud timer time is converted to UTC
    $timezone15minsTmp = DeviceHelper::ParseInt($jsonArr[9], -128, 127, 9);
    $device->Timezone = $timezone15minsTmp * 15;
    $device->IsTimezoneSet = $timezone15minsTmp != -128;

    $device->IntervalPowerMode = DeviceHelper::ParseInt($jsonArr[10], 0, 604800, 10);
    $device->IntervalBatteryMode = DeviceHelper::ParseInt($jsonArr[11], 0, 604800, 11);
    $device->RstOk = DeviceHelper::ParseBool($jsonArr[12], 12);
    $device->BufferedModeMaxMessagesCnt = DeviceHelper::ParseInt($jsonArr[13], 0, 255, 13);
    $device->AStateMask = DeviceHelper::ParseInt($jsonArr[14], 0, 65535, 14);
    $device->AState = DeviceHelper::ParseInt($jsonArr[15], 0, 65535, 15);
    $device->NConf = DeviceHelper::ParseInt($jsonArr[16], 0, 255, 16);
    $device->AlarmsEvalOffDueTimeLimit = DeviceHelper::ParseBool($jsonArr[17], 17);
    $device->CustomerUid = DeviceHelper::ParseString($jsonArr[18], 0, 16, 18);
    $device->BatteryPercentage = DeviceHelper::ParseInt($jsonArr[19], 0, 100, 19);
    $device->Channels = array();

    $chsIdx = 20;
    $jsonChs = $jsonArr[$chsIdx];
    DeviceHelper::CheckIsArray($jsonChs, 0, 8, $chsIdx);
    for ($i = 0; $i < count($jsonChs); $i++)
    {
      $jsonCh = $jsonChs[$i];
      DeviceHelper::CheckIsArray($jsonCh, 2, 9, $chsIdx, $i);

      $number = DeviceHelper::ParseInt($jsonCh[0], 1, 8, $chsIdx, $i, 0);
      $j = 0;
      $isDupl = false;
      while (($j < count($device->Channels)) && (!$isDupl))
      {
        if ($device->Channels[$j]->Number == $number)
          $isDupl = true;
        $j++;
      }
      if ($isDupl)
        throw new ParseException("", ERROR_CODE_DUPLICITE_CHANEL_NUMBER, $chsIdx, $i, 0);

      $isBin = DeviceHelper::parseBool($jsonCh[1], $chsIdx, $i, 1) == 1;
      if ($isBin)
        DeviceHelper::CheckIsArray($jsonCh, 7, 7, $chsIdx, $i);
      else
        DeviceHelper::CheckIsArray($jsonCh, 9, 9, $chsIdx, $i);
      $channel = $isBin ? $channel = new BinaryChannel() : $channel = new AnalogChannel();
      $channel->Number = $number;
      $channel->Quantity = DeviceHelper::ParseString($jsonCh[2], 0, 16, $chsIdx, $i, 2);
      if ($isBin)
      {
        DeviceHelper::CheckIsArray($jsonCh[3], 2, 2, $chsIdx, $i, 3);
        $channel->BinDescs0 = DeviceHelper::ParseString($jsonCh[3][0], 0, 16, $chsIdx, $i, 3, 0);
        $channel->BinDescs1 = DeviceHelper::ParseString($jsonCh[3][1], 0, 16, $chsIdx, $i, 3, 1);
        $channel->State = DeviceHelper::ParseBool($jsonCh[4], $chsIdx, $i, 4);
        DeviceHelper::CheckIsArray($jsonCh[5], 2, 2, $chsIdx, $i, 5);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[5][0], $chsIdx, $i, 5, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[5][1], $chsIdx, $i, 5, 1);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[6][0], 0, 2, $chsIdx, $i, 6, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[6][1], 0, 2, $chsIdx, $i, 6, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1ActiveOnStateInHigh = $tmp1 == 2 ? true : false;
        $channel->Alarm2ActiveOnStateInHigh = $tmp2 == 2 ? true : false;
      }
      else
      {
        $value = 0.0;
        $error = 0;
        $channel->IsValueCorrect = false;
        $channel->Value = 0.0;
        $channel->ValueError = 0;
        if (DeviceHelper::ParseHexFloat($jsonCh[3], $value, $error, $chsIdx, $i, 3))
        {
          $channel->Value = $value;
          $channel->IsValueCorrect = true;
        }
        else
          $channel->ValueError = $error;
        $channel->Unit = DeviceHelper::ParseString($jsonCh[4], 0, 8, $chsIdx, $i, 4);
        $channel->DecimalPlaces = DeviceHelper::ParseInt($jsonCh[5], 0, 10, $chsIdx, $i, 5);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[6][0], $chsIdx, $i, 6, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[6][1], $chsIdx, $i, 6, 1);
        DeviceHelper::CheckIsArray($jsonCh[7], 2, 2, $chsIdx, $i, 7);
        $channel->Alarm1Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][0], $chsIdx, $i, 7, 0);
        $channel->Alarm2Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][1], $chsIdx, $i, 7, 1);
        DeviceHelper::CheckIsArray($jsonCh[8], 2, 2, $chsIdx, $i, 8);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[8][0], 0, 2, $chsIdx, $i, 8, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[8][1], 0, 2, $chsIdx, $i, 8, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1IsHigherThenLimit = $tmp1 == 2 ? true : false;
        $channel->Alarm2IsHigherThenLimit = $tmp2 == 2 ? true : false;
      }
      array_push($device->Channels, $channel);
    }
    return $device;
  }
  private static function ParseType1Version6($device, $jsonArr)
  {
    DeviceHelper::CheckIsArray($jsonArr, 22, 22);
    $device->Rssi = DeviceHelper::ParseInt($jsonArr[2], 0, 99, 2);
    $device->TimeNow = DeviceHelper::ParseDatetime($jsonArr[3], 3);//JP: difference against comet cloud parser: here device time as is, in comet cloud timer time is converted to UTC
    $device->OrderId = DeviceHelper::ParseInt($jsonArr[4], 0, 255, 4);
    $device->IsAsync = DeviceHelper::ParseBool($jsonArr[5], 5);
    $device->Sn = DeviceHelper::ParseInt($jsonArr[6], 0, 99999999, 6);
    $device->Desc = DeviceHelper::ParseString($jsonArr[7], 0, 32, 7);
    $device->Kind = DeviceHelper::ParseInt($jsonArr[8], 0, 255, 8);

    $device->Time = DeviceHelper::ParseDatetime($jsonArr[9], 9);//JP: difference against comet cloud parser: here device time as is, in comet cloud timer time is converted to UTC
    $timezone15minsTmp = DeviceHelper::ParseInt($jsonArr[10], -128, 127, 10);
    $device->Timezone = $timezone15minsTmp * 15;
    $device->IsTimezoneSet = $timezone15minsTmp != -128;

    $device->IntervalPowerMode = DeviceHelper::ParseInt($jsonArr[11], 0, 604800, 11);
    $device->IntervalBatteryMode = DeviceHelper::ParseInt($jsonArr[12], 0, 604800, 12);
    $device->RstOk = DeviceHelper::ParseBool($jsonArr[13], 13);
    $device->BufferedModeMaxMessagesCnt = DeviceHelper::ParseInt($jsonArr[14], 0, 255, 14);
    $device->AStateMask = DeviceHelper::ParseInt($jsonArr[15], 0, 65535, 15);
    $device->AState = DeviceHelper::ParseInt($jsonArr[16], 0, 65535, 16);
    $device->NConf = DeviceHelper::ParseInt($jsonArr[17], 0, 255, 17);
    $device->AlarmsEvalOffDueTimeLimit = DeviceHelper::ParseBool($jsonArr[18], 18);
    $device->CustomerUid = DeviceHelper::ParseString($jsonArr[19], 0, 16, 19);
    $device->BatteryPercentage = DeviceHelper::ParseInt($jsonArr[20], 0, 100, 20);
    $device->Channels = array();

    $chsIdx = 21;
    $jsonChs = $jsonArr[$chsIdx];
    DeviceHelper::CheckIsArray($jsonChs, 0, 8, $chsIdx);
    for ($i = 0; $i < count($jsonChs); $i++)
    {
      $jsonCh = $jsonChs[$i];
      DeviceHelper::CheckIsArray($jsonCh, 2, 9, $chsIdx, $i);

      $number = DeviceHelper::ParseInt($jsonCh[0], 1, 8, $chsIdx, $i, 0);
      $j = 0;
      $isDupl = false;
      while (($j < count($device->Channels)) && (!$isDupl))
      {
        if ($device->Channels[$j]->Number == $number)
          $isDupl = true;
        $j++;
      }
      if ($isDupl)
        throw new ParseException("", ERROR_CODE_DUPLICITE_CHANEL_NUMBER, $chsIdx, $i, 0);

      $isBin = DeviceHelper::parseBool($jsonCh[1], $chsIdx, $i, 1) == 1;
      if ($isBin)
        DeviceHelper::CheckIsArray($jsonCh, 7, 7, $chsIdx, $i);
      else
        DeviceHelper::CheckIsArray($jsonCh, 9, 9, $chsIdx, $i);
      $channel = $isBin ? $channel = new BinaryChannel() : $channel = new AnalogChannel();
      $channel->Number = $number;
      $channel->Quantity = DeviceHelper::ParseString($jsonCh[2], 0, 16, $chsIdx, $i, 2);
      if ($isBin)
      {
        DeviceHelper::CheckIsArray($jsonCh[3], 2, 2, $chsIdx, $i, 3);
        $channel->BinDescs0 = DeviceHelper::ParseString($jsonCh[3][0], 0, 16, $chsIdx, $i, 3, 0);
        $channel->BinDescs1 = DeviceHelper::ParseString($jsonCh[3][1], 0, 16, $chsIdx, $i, 3, 1);
        $channel->State = DeviceHelper::ParseBool($jsonCh[4], $chsIdx, $i, 4);
        DeviceHelper::CheckIsArray($jsonCh[5], 2, 2, $chsIdx, $i, 5);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[5][0], $chsIdx, $i, 5, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[5][1], $chsIdx, $i, 5, 1);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[6][0], 0, 2, $chsIdx, $i, 6, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[6][1], 0, 2, $chsIdx, $i, 6, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1ActiveOnStateInHigh = $tmp1 == 2 ? true : false;
        $channel->Alarm2ActiveOnStateInHigh = $tmp2 == 2 ? true : false;
      }
      else
      {
        $value = 0.0;
        $error = 0;
        $channel->IsValueCorrect = false;
        $channel->Value = 0.0;
        $channel->ValueError = 0;
        if (DeviceHelper::ParseHexFloat($jsonCh[3], $value, $error, $chsIdx, $i, 3))
        {
          $channel->Value = $value;
          $channel->IsValueCorrect = true;
        }
        else
          $channel->ValueError = $error;
        $channel->Unit = DeviceHelper::ParseString($jsonCh[4], 0, 8, $chsIdx, $i, 4);
        $channel->DecimalPlaces = DeviceHelper::ParseInt($jsonCh[5], 0, 10, $chsIdx, $i, 5);
        DeviceHelper::CheckIsArray($jsonCh[6], 2, 2, $chsIdx, $i, 6);
        $channel->Alarm1 = DeviceHelper::ParseBool($jsonCh[6][0], $chsIdx, $i, 6, 0);
        $channel->Alarm2 = DeviceHelper::ParseBool($jsonCh[6][1], $chsIdx, $i, 6, 1);
        DeviceHelper::CheckIsArray($jsonCh[7], 2, 2, $chsIdx, $i, 7);
        $channel->Alarm1Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][0], $chsIdx, $i, 7, 0);
        $channel->Alarm2Limit = DeviceHelper::ParseHexFloatNoError($jsonCh[7][1], $chsIdx, $i, 7, 1);
        DeviceHelper::CheckIsArray($jsonCh[8], 2, 2, $chsIdx, $i, 8);
        $tmp1 = DeviceHelper::ParseInt($jsonCh[8][0], 0, 2, $chsIdx, $i, 8, 0);
        $tmp2 = DeviceHelper::ParseInt($jsonCh[8][1], 0, 2, $chsIdx, $i, 8, 1);
        $channel->Alarm1On = $tmp1 > 0 ? true : false;
        $channel->Alarm2On = $tmp2 > 0 ? true : false;
        $channel->Alarm1IsHigherThenLimit = $tmp1 == 2 ? true : false;
        $channel->Alarm2IsHigherThenLimit = $tmp2 == 2 ? true : false;
      }
      array_push($device->Channels, $channel);
    }
    return $device;
  }

  private static function CheckIsArray($input, $minLen, $maxLen, $i0 = -1, $i1 = -1, $i2 = -1, $i3 = -1)
  {
    if (!is_array($input))
      throw new ParseException("", ERROR_CODE_IS_NOT_ARRAY, $i0, $i1, $i2, $i3);
    if ((count($input) < $minLen) || (count($input) > $maxLen))
      throw new ParseException("", ERROR_CODE_ARRAY_OUT_OF_BOUNDS, $i0, $i1, $i2, $i3);
  }
  private static function ParseBool($input, $i0 = -1, $i1 = -1, $i2 = -1, $i3 = -1)
  {
    return DeviceHelper::ParseInt($input, 0, 1, $i0 = -1, $i1 = -1, $i2 = -1, $i3 = -1) == 1 ? true : false;
  }
  private static function ParseInt($input, $minVal, $maxVal, $i0 = -1, $i1 = -1, $i2 = -1, $i3 = -1)
  {
    if (!is_int($input))
      throw new ParseException("", ERROR_CODE_IS_NOT_INT, $i0, $i1, $i2, $i3);
    if (($input < $minVal) || ($input > $maxVal))
      throw new ParseException("", ERROR_CODE_INT_OUT_OF_BOUNDS, $i0, $i1, $i2, $i3);
    return $input;
  }
  private static function ParseString($input, $minLen, $maxLen, $i0 = -1, $i1 = -1, $i2 = -1, $i3 = -1)
  {
    if (!is_string($input))
      throw new ParseException("", ERROR_CODE_IS_NOT_STRING, $i0, $i1, $i2, $i3);
    if ((strlen($input) < $minLen) || (strlen($input) > $maxLen))
      throw new ParseException("", ERROR_CODE_STRING_OUT_OF_BOUNDS, $i0, $i1, $i2, $i3);
    return $input;
  }
  private static function ParseDatetime($input, $i0 = -1, $i1 = -1, $i2 = -1, $i3 = -1)
  {
    if (!is_string($input))
      throw new ParseException("", ERROR_CODE_IS_NOT_STRING, $i0, $i1, $i2, $i3);
    $pattern = '/^[a-fA-F0-9]{8,8}$/';
    if ( !(is_string($input) && preg_match($pattern, $input)) )
      throw new ParseException("", ERROR_CODE_DATETIME_BAD_FORMAT, $i0, $i1, $i2, $i3);
    return DateTime::createFromFormat(DateTime::ISO8601, "2000-01-01T00:00:00Z")->add(new DateInterval("PT".hexdec($input)."S"));
  }
  private static function ParseHexFloat($input, &$value, &$error, $i0 = -1, $i1 = -1, $i2 = -1, $i3 = -1)
  {
    $value = 0.0;
    $error = 0;
    if (!is_string($input))
      throw new ParseException("", ERROR_CODE_IS_NOT_STRING, $i0, $i1, $i2, $i3);
    //ve fw 0.0.90 je chyba, 0 se JSONU objevi jako "0" a ne jako hexastring proto nasl radek:
    if ($input == "0")
      return true;
    $pattern = '/^[a-fA-F0-9]{8,8}$/';
    if ( !(is_string($input) && preg_match($pattern, $input)) )
      throw new ParseException("", ERROR_CODE_FLOAT_BAD_HEX_FORMAT, $i0, $i1, $i2, $i3);
    return DeviceHelper::DecodeHexFloat($input, $value, $error);
  }
  private static function ParseHexFloatNoError($input, $i0 = -1, $i1 = -1, $i2 = -1, $i3 = -1)
  {
    $value = 0.0;
    $error = 0;
    if (!DeviceHelper::ParseHexFloat($input, $value, $error, $i0, $i1, $i2, $i3))
      throw new ParseException("", ERROR_CODE_FLOAT_ERROR_NOT_ALLOWED, $i0, $i1, $i2, $i3);
    return $value;
  }
  private static function Hex2float($number) {
    $binfinal = sprintf("%032b",hexdec($number));
    $sign = substr($binfinal, 0, 1);
    $exp = substr($binfinal, 1, 8);
    $mantissa = "1".substr($binfinal, 9);
    $mantissa = str_split($mantissa);
    $exp = bindec($exp)-127;
    $significand=0;
    for ($i = 0; $i < 24; $i++) {
      $significand += (1 / pow(2,$i))*$mantissa[$i];
    }
    return $significand * pow(2,$exp) * ($sign*-2+1);
  }
  private static function DecodeHexFloat($hex, &$value, &$error)
  {
    $dec = hexdec($hex);
    $value = 0.0;
    $error = 0;
    if (($dec == 0x7F800000) || ($dec == 0xFF800000) || ($dec == 0x7FFFFFFF))
    {
      $error = 90;
      return false;
    }
    if (($dec & 0xFFFF0000) == (0xFF810000 & 0xffffffff))
    {
      $error = $dec & 0xff;
      return false;
    }
    $value = DeviceHelper::Hex2float($hex);
    return true;
  }
  private static function AStateToString($astate)
  {
    return
      "  BinaryIn1 = "            . (($astate >>  0) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  BinaryIn2 = "            . (($astate >>  1) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  BinaryIn3 = "            . (($astate >>  2) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  BinaryIn4 = "            . (($astate >>  3) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  AcousticAlarmActive = "  . (($astate >>  4) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  OutRS232SignalActive = " . (($astate >>  5) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  ExternalPowerPresent = " . (($astate >>  6) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  RecordingActive = "      . (($astate >>  7) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  EmptyBattery = "         . (($astate >>  8) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  LowBattery = "           . (($astate >>  9) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  ExternalPowerError = "   . (($astate >> 10) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  MeasurementError = "     . (($astate >> 11) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  ConfigurationError = "   . (($astate >> 12) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  MemoryLimitExceeded = "  . (($astate >> 13) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  MemoryFull = "           . (($astate >> 14) & 1 == 1 ? "true" : "false") . "\r\n" .
      "  OpticalAlarmActive = "   . (($astate >> 15) & 1 == 1 ? "true" : "false") . "\r\n";
  }
  private static function AStateWithMaskToString($astatemask, $astate)
  {
    return
      "  BinaryIn1 =            " . (($astatemask >>  0) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  0) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  BinaryIn2 =            " . (($astatemask >>  1) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  1) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  BinaryIn3 =            " . (($astatemask >>  2) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  2) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  BinaryIn4 =            " . (($astatemask >>  3) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  3) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  AcousticAlarmActive =  " . (($astatemask >>  4) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  4) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  OutRS232SignalActive = " . (($astatemask >>  5) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  5) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  ExternalPowerPresent = " . (($astatemask >>  6) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  6) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  RecordingActive =      " . (($astatemask >>  7) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  7) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  EmptyBattery =         " . (($astatemask >>  8) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  8) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  LowBattery =           " . (($astatemask >>  9) & 1 == 1 ? "true " : "false") . ", " . (($astate >>  9) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  ExternalPowerError =   " . (($astatemask >> 10) & 1 == 1 ? "true " : "false") . ", " . (($astate >> 10) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  MeasurementError =     " . (($astatemask >> 11) & 1 == 1 ? "true " : "false") . ", " . (($astate >> 11) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  ConfigurationError =   " . (($astatemask >> 12) & 1 == 1 ? "true " : "false") . ", " . (($astate >> 12) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  MemoryLimitExceeded =  " . (($astatemask >> 13) & 1 == 1 ? "true " : "false") . ", " . (($astate >> 13) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  MemoryFull =           " . (($astatemask >> 14) & 1 == 1 ? "true " : "false") . ", " . (($astate >> 14) & 1 == 1 ? "true " : "false") . "\r\n" .
      "  OpticalAlarmActive =   " . (($astatemask >> 15) & 1 == 1 ? "true " : "false") . ", " . (($astate >> 15) & 1 == 1 ? "true " : "false") . "\r\n";
  }

}  